<?php

namespace App\Http\Controllers\Frontend;

use App\Http\Controllers\Controller;
use App\Models\Post;
use App\Models\PostCategory;
use App\Models\Tag;
use Illuminate\Http\Request;
use Illuminate\View\View;

class BlogController extends Controller
{
    public function index(Request $request): View
    {
        $categories = PostCategory::active()
            ->withCount('posts')
            ->orderBy('order')
            ->get();

        $query = Post::published()
            ->with(['author', 'category', 'tags'])
            ->orderByDesc('published_at');

        if ($request->filled('category')) {
            $query->whereHas('category', function ($q) use ($request) {
                $q->where('slug', $request->category);
            });
        }

        if ($request->filled('tag')) {
            $query->whereHas('tags', function ($q) use ($request) {
                $q->where('slug', $request->tag);
            });
        }

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('title', 'like', "%{$search}%")
                    ->orWhere('content', 'like', "%{$search}%");
            });
        }

        $posts = $query->paginate(9);

        $featuredPosts = Post::published()
            ->featured()
            ->with(['author', 'category'])
            ->limit(3)
            ->get();

        $popularTags = Tag::withCount('posts')
            ->orderByDesc('posts_count')
            ->limit(10)
            ->get();

        return view('frontend.pages.blog.index', compact(
            'posts',
            'categories',
            'featuredPosts',
            'popularTags'
        ));
    }

    public function show(string $slug): View
    {
        $post = Post::with(['author', 'category', 'tags'])
            ->where('slug', $slug)
            ->published()
            ->firstOrFail();

        $post->incrementViews();

        $relatedPosts = Post::published()
            ->where('id', '!=', $post->id)
            ->when($post->category_id, function ($q) use ($post) {
                $q->where('category_id', $post->category_id);
            })
            ->limit(3)
            ->get();

        return view('frontend.pages.blog.show', compact('post', 'relatedPosts'));
    }

    public function category(string $slug): View
    {
        $category = PostCategory::where('slug', $slug)
            ->active()
            ->firstOrFail();

        $posts = $category->publishedPosts()
            ->with(['author', 'tags'])
            ->paginate(9);

        return view('frontend.pages.blog.category', compact('category', 'posts'));
    }

    public function tag(string $slug): View
    {
        $tag = Tag::where('slug', $slug)->firstOrFail();

        $posts = $tag->posts()
            ->published()
            ->with(['author', 'category'])
            ->paginate(9);

        return view('frontend.pages.blog.tag', compact('tag', 'posts'));
    }
}
