<?php

namespace App\Http\Controllers\Frontend;

use App\Http\Controllers\Controller;
use App\Http\Requests\ContactFormRequest;
use App\Http\Requests\QuoteFormRequest;
use App\Models\Lead;
use App\Models\Setting;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Facades\Mail;
use Illuminate\View\View;

class ContactController extends Controller
{
    public function index(): View
    {
        return view('frontend.pages.contact');
    }

    public function store(ContactFormRequest $request): RedirectResponse
    {
        $lead = Lead::create([
            'type' => 'contact',
            'name' => $request->name,
            'email' => $request->email,
            'phone' => $request->phone,
            'company' => $request->company,
            'subject' => $request->subject,
            'message' => $request->message,
            'source_page' => url()->previous(),
            'ip_address' => $request->ip(),
            'user_agent' => $request->userAgent(),
        ]);

        // Send notification email
        $this->sendNotificationEmail($lead, 'contact');

        return back()->with('success', __('Thank you for contacting us. We will get back to you soon.'));
    }

    public function quote(): View
    {
        return view('frontend.pages.quote');
    }

    public function storeQuote(QuoteFormRequest $request): RedirectResponse
    {
        $lead = Lead::create([
            'type' => 'quote',
            'name' => $request->name,
            'email' => $request->email,
            'phone' => $request->phone,
            'company' => $request->company,
            'subject' => 'Quote Request',
            'message' => $request->requirements,
            'data' => [
                'industry' => $request->industry,
                'service' => $request->service,
                'quantity' => $request->quantity,
                'timeline' => $request->timeline,
                'budget' => $request->budget,
            ],
            'source_page' => url()->previous(),
            'ip_address' => $request->ip(),
            'user_agent' => $request->userAgent(),
        ]);

        // Handle file upload if present
        if ($request->hasFile('attachment')) {
            $path = $request->file('attachment')->store('quote-attachments', 'public');
            $lead->update(['data->attachment' => $path]);
        }

        $this->sendNotificationEmail($lead, 'quote');

        return back()->with('success', __('Thank you for your quote request. Our team will contact you within 24 hours.'));
    }

    protected function sendNotificationEmail(Lead $lead, string $type): void
    {
        $recipientKey = $type === 'quote' ? 'quote_form_email' : 'contact_form_email';
        $recipient = Setting::get($recipientKey, config('mail.from.address'));

        // In production, send actual email
        // Mail::to($recipient)->send(new \App\Mail\LeadNotification($lead));
    }
}
