<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Spatie\Sluggable\HasSlug;
use Spatie\Sluggable\SlugOptions;

class Job extends Model
{
    use HasSlug;

    protected $fillable = [
        'title',
        'title_ar',
        'slug',
        'department',
        'location',
        'type',
        'description',
        'description_ar',
        'requirements',
        'requirements_ar',
        'benefits',
        'benefits_ar',
        'salary_range',
        'is_published',
        'deadline',
        'order',
    ];

    protected $casts = [
        'is_published' => 'boolean',
        'deadline' => 'date',
    ];

    public const TYPES = [
        'full-time' => 'Full Time',
        'part-time' => 'Part Time',
        'contract' => 'Contract',
        'internship' => 'Internship',
    ];

    public function getSlugOptions(): SlugOptions
    {
        return SlugOptions::create()
            ->generateSlugsFrom('title')
            ->saveSlugsTo('slug');
    }

    public function applications(): HasMany
    {
        return $this->hasMany(JobApplication::class);
    }

    public function scopePublished($query)
    {
        return $query->where('is_published', true);
    }

    public function scopeActive($query)
    {
        return $query->published()
            ->where(function ($q) {
                $q->whereNull('deadline')
                    ->orWhere('deadline', '>=', now());
            });
    }

    public function getLocalizedTitle(): string
    {
        return app()->getLocale() === 'ar' && $this->title_ar ? $this->title_ar : $this->title;
    }

    public function getLocalizedDescription(): ?string
    {
        return app()->getLocale() === 'ar' && $this->description_ar ? $this->description_ar : $this->description;
    }

    public function getLocalizedRequirements(): ?string
    {
        return app()->getLocale() === 'ar' && $this->requirements_ar ? $this->requirements_ar : $this->requirements;
    }

    public function getLocalizedBenefits(): ?string
    {
        return app()->getLocale() === 'ar' && $this->benefits_ar ? $this->benefits_ar : $this->benefits;
    }

    public function getTypeLabelAttribute(): string
    {
        return self::TYPES[$this->type] ?? $this->type;
    }
}
